<?php
/**
 * External API client for AnalyticReporting
 *
 * Class intended to provide simple functions for
 * accessing external API methods
 *
 */
class SuiteAPIClient  extends AbstractAPIClient
{
    private $info; // Curl info

    /**
     * @var $errors array
     *
     * Array of error strings
     */
    public $errors = array();


    public $response;
    public $headers = array();
    public $body;

    public $progress = false;

    public $method;

    /** @var bool status of CURL timeout */
    protected $timedOut = false;

    /**
     * Class options, for example CURL options
     *
     * @var array
     */
    protected $options = array(
        "curlMaxTimeout" => 1500,
    );

    public function __construct()
    {
        // Operations with outside server could take long time
        FunctionWrapper::_set_time_limit(0);
    }


    /**
     * Set class option one by one
     *
     * @param $option
     * @param $value
     * @return bool
     */
    public function setOption($option, $value)
    {
        // Return false if empty option/value
        if(empty($option) || empty($value)) {
            return false;
        }

        $this->options[$option] = $value;
        return true;
    }

    public function setResponse($response)
    {
        $this->response = $response;
    }

    public function getResponse()
    {
        return $this->response;
    }

    public function setBody($body)
    {
        $this->body = $body;
    }

    public function getBody()
    {
        return $this->body;
    }

    public function setHeaders($headers = array())
    {
        $this->headers = $headers;
    }

    public function getHeaders()
    {
        return $this->headers;
    }

    public function getHeader($header)
    {
        return $this->headers[$header];
    }

    public function setMethod($method)
    {
        $this->method = $method;
    }

    public function getMethod()
    {
        return $this->method;
    }

    public function getErrors()
    {
        return $this->errors;
    }

    /**
     * Set CURL_INFO before closing connection
     *
     * @param $info
     */
    public function setInfo($info)
    {
        $this->info = $info;
    }

    /**
     * Get information regarding a specific transfer
     *
     * @return mixed
     */
    public function getInfo()
    {
        return $this->info;
    }

    /**
     * Set status of CURL timeout
     *
     * @param $timedOut
     */
    protected function setTimedOut($timedOut)
    {
        $this->timedOut = $timedOut;
    }

    /**
     * Return true if timedOut or false if not
     *
     * @return bool
     */
    public function getTimedOut()
    {
        return (bool) $this->timedOut;
    }

    /**
     * Check if object instance have any errors
     *
     * @return bool
     */
    public function haveErrors()
    {
        // Check if errors are set and not empty strings
        if((count($this->errors)) > 0 && (!empty($this->errors[0]))) {
            return true;
        }

        return false;
    }

    /**
     * Try to decide if response have file or not
     * File is/should be passed only if Content-Disposition header is set
     *
     * @return bool
     */
    public function isFile()
    {
        $key = 'Content-Disposition';
        $headers = $this->getHeaders();
        if(!empty($headers[$key])) {
            return true;
        }

        return false;
    }

    /**
     * @param $url string API URL use for data transfer
     * @param $data array Data array of cols and data elements
     * @param array $headers
     *
     * Post data array through CURL and receive file
     */
    public function fetch($url, $data = array(), $headers = array())
    {
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);

        if($this->getMethod() == self::HTTP_METHOD_POST) {
            $postFields = http_build_query($data);
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_POSTFIELDS, $postFields);
        }

        curl_setopt($ch, CURLOPT_VERBOSE, 1);
        curl_setopt($ch, CURLOPT_HEADER, 1);
        curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
        curl_setopt($ch, CURLOPT_BINARYTRANSFER, 1);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
        // The number of seconds to wait while trying to connect. Use 0 to wait indefinitely.
        curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 0);
        // The maximum number of seconds to allow cURL functions to execute.
        curl_setopt($ch, CURLOPT_TIMEOUT, $this->options["curlMaxTimeout"]);
        curl_setopt($ch, CURLOPT_FAILONERROR, true);
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);

        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        // CURLOPT_SSL_VERIFYHOST no longer accepts the value 1, value 2 will be used instead
        curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);

        // CURLOPT_SSL_VERIFYSTATUS added with PHP7 (on older PHP versions can broke system with errors)
        if(defined("CURLOPT_SSL_VERIFYSTATUS")) {
            curl_setopt($ch, CURLOPT_SSL_VERIFYSTATUS, false);
        }
        curl_setopt($ch, CURLOPT_ENCODING, '');

        $response = curl_exec($ch);
        $this->setStatusCode(curl_getinfo($ch, CURLINFO_HTTP_CODE));
        $this->setResponse($response);
        $errorNo = curl_errno($ch);
        $errorMessage = curl_error($ch);
        $this->errors[] = $errorNo;
        $this->errors[] = $errorMessage;

        $headerSize = curl_getinfo($ch, CURLINFO_HEADER_SIZE);
        $this->setInfo(curl_getinfo($ch));
        curl_close ($ch);

        // CURL has timed out
        if (in_array($errorNo, array(CURLE_OPERATION_TIMEDOUT, CURLE_OPERATION_TIMEOUTED))) {
            $this->setTimedOut(true);
            throw new Exception($errorMessage);
        }

        // Get headers and body
        $header = substr($response, 0, $headerSize);
        $body = substr($response, $headerSize);

        // Parse headers into key/value array
        $responseHeaders = array();
        $headerLines = explode("\n", $header);
        foreach($headerLines as $line) {
            $headerParts = explode(':', $line);
            if(count($headerParts) == 2) {
                $responseHeaders[$headerParts[0]] = trim($headerParts[1]);
            }
        }

        // Sometimes, headers can be empty
        $this->setHeaders($responseHeaders);
        $this->setBody($body);
    }

    /**
     * @param $url string API URL use for data transfer
     * @param $progressCallback bool NOT IMPLEMENTED JET
     *
     * Retrieve JSON with GET method from URL
     */
    public function retrieve($url, $progressCallback = false)
    {
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
        // The number of seconds to wait while trying to connect. Use 0 to wait indefinitely.
        curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 0);
        // The maximum number of seconds to allow cURL functions to execute.
        curl_setopt($ch, CURLOPT_TIMEOUT, $this->options["curlMaxTimeout"]);
        curl_setopt($ch, CURLOPT_FAILONERROR, true);
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
        curl_setopt($ch, CURLOPT_ENCODING, '');

        // Add progress callback method
        if($progressCallback) {
            curl_setopt($ch, CURLOPT_NOPROGRESS, false );
            curl_setopt($ch, CURLOPT_PROGRESSFUNCTION, $progressCallback);
        }

        $response = curl_exec($ch);
        $this->setResponse($response);
        $this->setBody($response);
        $this->errors[] = curl_errno($ch);
        $this->errors[] = curl_error($ch);

        $this->setInfo(curl_getinfo($ch));
        curl_close($ch);
    }

    /**
     * Retrieve URL contents with GET method
     *
     * @param $url
     * @return mixed
     * @throws Exception
     */
    public function getUrl($url)
    {
        if ($this->urlNotFound($url)) {
            throw new \Exception("Url `{$url}` doesn't exist");
        }

        $this->fetch($url);

        return $this->asJson();
    }

    public function urlNotFound($url) {
        $url = filter_var($url, FILTER_VALIDATE_URL);

        /* Open curl connection */
        $handle = curl_init($url);

        /* Set curl parameter */
        curl_setopt_array($handle, array(
            CURLOPT_FOLLOWLOCATION => TRUE,     // we need the last redirected url
            CURLOPT_NOBODY => TRUE,             // we don't need body
            CURLOPT_HEADER => FALSE,            // we don't need headers
            CURLOPT_RETURNTRANSFER => FALSE,    // we don't need return transfer
            CURLOPT_SSL_VERIFYHOST => FALSE,    // we don't need verify host
            CURLOPT_SSL_VERIFYPEER => FALSE     // we don't need verify peer
        ));

        /* Get the HTML or whatever is linked in $url. */
        $response = curl_exec($handle);

        $httpCode = curl_getinfo($handle, CURLINFO_EFFECTIVE_URL);  // Try to get the last url
        $httpCode = curl_getinfo($handle, CURLINFO_HTTP_CODE);      // Get http status from last url

        /* Close curl connection */
        curl_close($handle);

        return $httpCode === 404;
    }
}